//*********************************************************************************************************
//   Program written for Arduino Nano to drive Vishay "AGC 0" type fast proximity sensor consisting of TSSP4056
//   sensor (connected to pin D8) and VSLB3940 IR emitter (connected with a series resistor to pin D3).
//   Author: John Fisher
//*********************************************************************************************************
//   Version 2. The author wishes to thank AnalysIR (www.AnalysIR.com) for their many helpful improvements included
//   in this version, and especially for finding the "glitch" where under some conditions the Tx pulse repetition
//   rate was too fast and did not allow the Rx to turn off. Baud rate defaults to 250000, must set in serial monitor.
//*********************************************************************************************************

#define ANALYSIR false                                          // set true to use Arduino serial & false to use original direct access to UART
volatile uint8_t BurstCount = 25;                               // global variable for number of carrier cycles in burst
volatile uint16_t InputCapture = 0;                             // used to capture the sensor pulse-length from 16 bit timer1
volatile enum {rx_IDLE, rx_PENDING, rx_DONE} rx_stat = rx_IDLE; // rx_IDLE = no pulse yet seen, rx_PENDING= Rx is capturing a pulse, rx_DONE = Rx finished capturing the pulse
volatile enum {tx_PENDING, tx_DONE} tx_stat = tx_DONE;          // tx_PENDING = tx is sending a burst, tx_DONE = tx finished sending a burst


//*********************************************************************************************************
//  Interrupt service routine
//  Timer2 overflow interrupt used to count carrier cycles
//*********************************************************************************************************
ISR(TIMER2_OVF_vect) {
  static uint16_t carrier_count = 0;       // persistent counter used to count carrier cycles
  carrier_count++;                         // increment carrier cycle counter
  if (carrier_count >= BurstCount) {       // if burst is completed
    carrier_count = 0;                     // reset counter for start of new cycle
    TCCR2B &= 0B11111000;                  // clear the CS bits, which halts Timer2
    TCCR2A = 0;                            // return pin D3 to normal IO mode for now
    PORTD &= ~(1 << DDD3);                 // turn off emitter when idle (until next burst)
    tx_stat = tx_DONE;                     // change tx status to finished sending a burst
  }                                        // end if
}                                          // end ISR

//*********************************************************************************************************
//  Interrupt service routine
//  Timer1 edge based interrupts to capture the sensor pulse
//*********************************************************************************************************
ISR(TIMER1_CAPT_vect) {
  uint8_t scratch = TCCR1B & (1 << ICES1); // result should be 64 if set to trigger on rising edge, 0 if set to trigger on falling edge
  switch (scratch) {                       // switch depending on the ICES1 bit
    case 0:                                // ICES1 = 0, have falling edge trigger
      TCNT1 = 0;                           // reset Timer1
      TCCR1B |= (1 << ICES1);              // set interrupt to rising edge trigger
      rx_stat = rx_PENDING;                // set status to the Rx is capturing a pulse
      break;
    case 64:                               // ICES1 = 1, have rising edge trigger
      InputCapture = ICR1;                 // save the capture register value
      TCCR1B &= ~(1 << ICES1);             // set interrupt to falling edge trigger
      rx_stat = rx_DONE;                   // set status to the Rx has finished capturing a pulse
      break;
  }                                        // end switch
}                                          // end ISR


//**************************************************************************************************************************
//  Routine to send a single pulse. Function arguments: half-period of the carrier, used in Timer2 = (1/2xburstfreq) / (1/clkfreq), the
//  duty cycle of the carrier in %, and the number of cycles in a burst. The function returns the width of the sensor pulse from Timer1,
//  or -1 if no reflection is seen, or 2 if an invalid reflection is seen, i.e. outside the receiver spec.
//****************************************************************************************************************************
int16_t SendBurst (uint8_t HalfPeriod = 210, uint8_t BurstLength = 25, uint8_t Duty = 50) {

  int16_t Timer1Count;                        // holds the return value
  
  // -------------------------------------  set up timer 2 for transmitter --------------
  BurstCount = BurstLength;                   // set global variable visible to Timer2 ISR
  TCCR2A = (1 << WGM20) | (1 << COM2B1);      // clear OC2B on match counting up and set counting down, Mode5: PWM phase correct with TOP=OCR2A
  TCCR2B = (1 << WGM22);                      // FOC not used, no pre-scaler, final bit of Mode5 setting, cs bits are all zero so timer is off
  TIMSK2 = (1 << TOIE2);                      // enable timer2 overflow interrupt
  OCR2A  = HalfPeriod;                        // set timer2 TOP value
  OCR2B  = HalfPeriod * Duty / 100;           // set switching value for the output pin
  TCNT2 = 0;                                  // reset Timer2 to zero

  // -------------------------------------  set up timer 1 for receiver -----------------
  uint16_t PulsTolHi = (BurstLength + 6) * HalfPeriod / 32; // output pulse upper tolerance, see datasheet fig. 1
  uint16_t PulsTolLo = (BurstLength - 15) * HalfPeriod / 32; // output pulse lower tolerance, wider than datasheet to account for off BP frequencies
  TCCR1A = 0;                                 // normal mode, no output pins connected, TOP=0xFFFF
  rx_stat = rx_IDLE;                          // set the receiver state to idle
  TCCR1B = (1 << ICNC1) | (0 << ICES1) | (1 << CS11) | (1 << CS10); // noise cancelling input on, input capture edge select to falling edge, prescaler 64
  TIMSK1 = (1 << ICIE1);                      // enable timer1 input capture interrupt (transfers TCNT1 to ICR1 upon the edge defined)

  // -------------------------------------  end set up, start the transmission -----------------------------
  tx_stat = tx_PENDING;                       // tx status is sending burst
  TIFR1 |= (1 << ICF1);                       // make sure any pending interrupt is cleared...potential noise or interference
  TIFR2 |= (1 << TOV2);                       // make sure any pending interrupt is cleared
  sei();                                      // allow interrupts
  TCCR2B |= (1 << CS20);                      // start Timer2 with no pre-scale
  while (tx_stat != tx_DONE )                 // if the transmitted burst has not been completely sent
    ;                                         // normally do nothing until tx_stat = tx_DONE
  switch (rx_stat) {                          // check status of receiver
    case rx_IDLE:                             // if receiver is idle, no reflection was seen at the highest sensitivity, assume that object is out of range
      Timer1Count = -1;                       // -1 means no reflection
      break;
    case rx_PENDING:                          // if receiver is pending, we saw a reflection and we are currently measuring it
      while (rx_stat == rx_PENDING)           // can do nothing useful so wait for it to end
        ;                                     // do nothing until rx_stat = rx_DONE
    //break;                                  // no break here! once rx_stat != rx_PENDING, we want to fall through to the next case, not exit the switch!!
    case rx_DONE:                             // if receiver is done, the reflected pulse was measured, now check its validity
      Timer1Count = InputCapture;             // store the timer1 value
      rx_stat = rx_IDLE;                      // rx finished, set status to idle
      if ( !((Timer1Count > PulsTolLo) && (Timer1Count < PulsTolHi))) { // if we received an out of spec pulse
        Timer1Count = 2;                      // return magic number 2 to signal out of spec pulse
      }
      break;                                  // this is the end of the case: rx_stat = rx_IDLE
  }
  cli();                                      // clear interrupts
  return Timer1Count;
}                                             // end SendBurst

//***********************************************************************************************************
//  main Arduino setup() and loop() routines
//***********************************************************************************************************
//#define BAUD_RATE 57600
//#define BAUD_RATE 115200
//#define BAUD_RATE 200000
#define BAUD_RATE 250000
#define BAUD_RATE_DIVISOR (((F_CPU / 16)+BAUD_RATE/2)/ BAUD_RATE - 1) //added by AnalysIR rounding for integer division

void usart_putc(char c) {
  loop_until_bit_is_set(UCSR0A, UDRE0);       // wait for transmit buffer to be empty
  UDR0 = c;                                   // transmit character on the serial port
  return;
}

void setup() {
  cli();                                      // disable interrupts until we need them

  #if ANALYSIR                                // initialize the serial port
    Serial.begin(BAUD_RATE);
  #else
    UCSR0A = 0;
    UCSR0B = 1 << TXEN0;
    UCSR0C = 1 << UCSZ01 | 1 << UCSZ00;
    UBRR0 = BAUD_RATE_DIVISOR;
  #endif

  DDRD |= (1 << DDD3);
  PORTD &= ~(1 << DDD3);                                      // turn off emitter until needed later
  DDRB &= ~(1 << DDB0);                                       // define Arduino D8, AVR PB0 as input
  if ((PORTB & (1 << PORTB0)) == 0)   PORTB |= (1 << PORTB0); // this activates the internal pull-up resistor in case the IR sensor is disconnected

}    // end setup

void loop() {
  while (true) {                                              // avoids extra processing at end of every loop
    int16_t pulsewidth = 0;                                   // variable to hold the return value from function SendBurst
    uint8_t HalfPeriodHi = 250;                               // Timer2 value for longest period (lowest frequency e.g. 32kHz) (1/2xModFrequency in kHz) / Period_16MHz: e.g. 54kHz=148, 38kHz=210, 32kHz=250
    uint8_t HalfPeriodLo = 148;                               // Timer2 value for shortest period (highest frequency e.g. 54kHz)
    uint8_t HalfPeriodTry = HalfPeriodLo;                     // Timer2 value for next try (first try @ max sensitivity to detect when not idle)
    uint8_t Approx = 8;                                       // number of successive approximations

    while (pulsewidth  <= 2) {                                // if no reflection or bad reflection
      pulsewidth = SendBurst(HalfPeriodTry);
      delayMicroseconds(150);                                 // allow time for receiver to respond(go idle), before next try
    }                                                         // then keep trying

    while (Approx-- != 0) {
      HalfPeriodTry = (HalfPeriodHi + HalfPeriodLo) / 2 ;     // calculate the new half period for HalfPeriodTry
      if ((pulsewidth = SendBurst(HalfPeriodTry)) <= 2)       // if no reflection or bad reflection
        HalfPeriodHi = HalfPeriodTry;                         // raise the sensitivity adjustment by half
      else                                                    // if good reflection
        HalfPeriodLo = HalfPeriodTry;                         // lower the sensitivity adjustment by half
    }                                                         // end while

  uint8_t i = (HalfPeriodTry - 147);                          // normalize the result to start at zero

  #if ANALYSIR                                                // print string of '*' corresponding to the value in HalfPeriodTry
    while (i-- != 0)
      Serial.write('*');
    Serial.write('\r');
    Serial.write('\n');

  #else
    while (i-- != 0)
      usart_putc('*');
    usart_putc('\r');                                         // added  by AnalysIR - helps when using other terminal apps that need CRLF
    usart_putc('\n');
  #endif

  }                                                           // end while true

}                                                             // end loop

